// Package config 定义了虚拟Agent的配置结构
// 包含系统信息、硬件规格、负载模拟等配置项
package config

import (
	"fmt"
)

// SimulationMode 仿真模式
type SimulationMode string

const (
	// SimulationFixed 固定值模式
	SimulationFixed SimulationMode = "fixed"
	// SimulationLinear 线性变化模式
	SimulationLinear SimulationMode = "linear"
	// SimulationRandom 范围随机模式
	SimulationRandom SimulationMode = "random"
	// SimulationWave 波形变化模式
	SimulationWave SimulationMode = "wave"
)

// CPUConfig CPU配置
type CPUConfig struct {
	Model string      `json:"model"` // CPU型号
	Arch  string      `json:"arch"`  // 架构
	Cores int         `json:"cores"` // 核心数
	Usage UsageConfig `json:"usage"` // CPU使用率配置
}

// MemoryConfig 内存配置
type MemoryConfig struct {
	Total uint64      `json:"total"` // 总内存(字节)
	Usage UsageConfig `json:"usage"` // 内存使用率配置
}

// DiskConfig 磁盘配置
type DiskConfig struct {
	Total uint64      `json:"total"` // 总磁盘空间(字节)
	Usage UsageConfig `json:"usage"` // 磁盘使用率配置
}

// SwapConfig 交换分区配置
type SwapConfig struct {
	Total uint64      `json:"total"` // 总交换分区(字节)
	Usage UsageConfig `json:"usage"` // 交换分区使用率配置
}

// NetworkConfig 网络配置
type NetworkConfig struct {
	InSpeed  UsageConfig `json:"in_speed"`  // 入站速度配置
	OutSpeed UsageConfig `json:"out_speed"` // 出站速度配置
	InTotal  UsageConfig `json:"in_total"`  // 入站总量配置
	OutTotal UsageConfig `json:"out_total"` // 出站总量配置
}

// LoadConfig 负载配置
type LoadConfig struct {
	Load1  UsageConfig `json:"load1"`  // 1分钟负载
	Load5  UsageConfig `json:"load5"`  // 5分钟负载
	Load15 UsageConfig `json:"load15"` // 15分钟负载
}

// ProcessConfig 进程配置
type ProcessConfig struct {
	Count   UsageConfig `json:"count"`    // 进程数量
	TcpConn UsageConfig `json:"tcp_conn"` // TCP连接数
	UdpConn UsageConfig `json:"udp_conn"` // UDP连接数
}

// GPUConfig GPU配置
type GPUConfig struct {
	Models []string    `json:"models"` // GPU型号列表
	Usage  UsageConfig `json:"usage"`  // GPU使用率配置
}

// UsageConfig 使用率配置
type UsageConfig struct {
	Mode     SimulationMode `json:"mode"`     // 仿真模式
	Min      float64        `json:"min"`      // 最小值
	Max      float64        `json:"max"`      // 最大值
	Base     float64        `json:"base"`     // 基础值(用于固定模式)
	Step     float64        `json:"step"`     // 步长(用于线性模式)
	Period   int            `json:"period"`   // 周期(用于波形和季节模式)
	Variance float64        `json:"variance"` // 方差(用于随机模式)
}

// SystemConfig 系统信息配置
type SystemConfig struct {
	Platform        string `json:"platform"`         // 操作系统平台
	PlatformVersion string `json:"platform_version"` // 操作系统版本
	Arch            string `json:"arch"`             // 系统架构
	Virtualization  string `json:"virtualization"`   // 虚拟化类型
	Version         string `json:"version"`          // 版本号
}

// ServerConfig 服务器连接配置
type ServerConfig struct {
	Address      string `json:"address"`       // 服务器地址
	ClientSecret string `json:"client_secret"` // 客户端密钥
	ClientUUID   string `json:"client_uuid"`   // 客户端UUID
	Interval     int    `json:"interval"`      // 上报间隔(毫秒)
}

// AgentConfig agent配置
type AgentConfig struct {
	System  SystemConfig  `json:"system"`  // 系统信息配置
	CPU     CPUConfig     `json:"cpu"`     // CPU配置
	Memory  MemoryConfig  `json:"memory"`  // 内存配置
	Disk    DiskConfig    `json:"disk"`    // 磁盘配置
	Swap    SwapConfig    `json:"swap"`    // 交换分区配置
	Network NetworkConfig `json:"network"` // 网络配置
	Load    LoadConfig    `json:"load"`    // 负载配置
	Process ProcessConfig `json:"process"` // 进程配置
	GPU     GPUConfig     `json:"gpu"`     // GPU配置
	Server  ServerConfig  `json:"server"`  // 服务器配置
}

// GetCPUInfo 获取CPU信息字符串
func (c *AgentConfig) GetCPUInfo() []string {
	cpuInfo := fmt.Sprintf("%s %d Core", c.CPU.Model, c.CPU.Cores)
	return []string{cpuInfo}
}
