package agent

import (
	"fmt"
	"time"
	"virtual-nezha-agent/config"
	"virtual-nezha-agent/simulator"
)

func NewInstance(req CreateRequest) (*Instance, error) {
	if req.ServerAddress == "" || req.ClientSecret == "" || req.ClientUUID == "" {
		return nil, fmt.Errorf("server_address, client_secret, and client_uuid are required")
	}
	if req.CPU <= 0 || req.RAM <= 0 || req.DiskSize <= 0 {
		return nil, fmt.Errorf("cpu, ram, and disk_size must be greater than 0")
	}
	if req.CountryCode == "" {
		return nil, fmt.Errorf("country_code is required")
	}
	if req.CPUModel == "" {
		return nil, fmt.Errorf("cpu_model is required")
	}

	if req.LoadLevel == "" {
		req.LoadLevel = "medium"
	}
	if req.LoadLevel != "low" && req.LoadLevel != "medium" && req.LoadLevel != "high" {
		return nil, fmt.Errorf("invalid load_level, must be one of: low, medium, high")
	}

	agent := &Instance{
		ID:            req.ClientUUID,
		ServerAddress: req.ServerAddress,
		ClientSecret:  req.ClientSecret,
		ClientUUID:    req.ClientUUID,
		CountryCode:   req.CountryCode,
		CPUModel:      req.CPUModel,
		CPU:           req.CPU,
		RAM:           req.RAM,
		Swap:          req.Swap,
		DiskSize:      req.DiskSize,
		LoadLevel:     req.LoadLevel,
		Status:        "starting",
		UseSSL:        req.UseSSL,
		CreatedAt:     time.Now(),
		UpdatedAt:     time.Now(),
		ConnectionStatus: ConnectionStatus{
			Status: ConnectionStatusIdle,
		},
		reconnectConfig: DefaultReconnectConfig,
		healthCheckStop: make(chan struct{}),
	}

	agent.config = agent.createAgentConfig()
	agent.simulator = simulator.NewDataSimulator(agent.config)

	return agent, nil
}

func (a *Instance) createAgentConfig() *config.AgentConfig {
	cpuUsage, memUsage, diskUsage := GetLoadLevelConfig(a.LoadLevel)
	networkConfig := GetNetworkLoadLevelConfig(a.LoadLevel)
	loadConfig := GetSystemLoadLevelConfig(a.LoadLevel)

	return &config.AgentConfig{
		System: config.SystemConfig{
			Platform:        "debian",
			PlatformVersion: "12",
			Arch:            "x86_64",
			Version:         "1.13.0",
		},
		CPU: config.CPUConfig{
			Model: a.CPUModel,
			Arch:  "x86_64",
			Cores: a.CPU,
			Usage: cpuUsage,
		},
		Memory: config.MemoryConfig{
			Total: uint64(a.RAM) * 1024 * 1024,
			Usage: memUsage,
		},
		Disk: config.DiskConfig{
			Total: uint64(a.DiskSize) * 1024 * 1024 * 1024,
			Usage: diskUsage,
		},
		Swap: config.SwapConfig{
			Total: uint64(a.Swap) * 1024 * 1024,
			Usage: config.UsageConfig{
				Mode:     config.SimulationFixed,
				Base:     5.0,
				Variance: 2.0,
			},
		},
		Network: networkConfig,
		Load:    loadConfig,
		Process: config.ProcessConfig{
			Count: config.UsageConfig{
				Mode:     config.SimulationRandom,
				Min:      80,
				Max:      150,
				Variance: 20,
			},
			TcpConn: config.UsageConfig{
				Mode:     config.SimulationRandom,
				Min:      20,
				Max:      100,
				Variance: 15,
			},
			UdpConn: config.UsageConfig{
				Mode:     config.SimulationRandom,
				Min:      5,
				Max:      30,
				Variance: 8,
			},
		},
		GPU: config.GPUConfig{
			Models: []string{},
			Usage: config.UsageConfig{
				Mode: config.SimulationFixed,
				Base: 0,
			},
		},
		Server: config.ServerConfig{
			Address:      a.ServerAddress,
			ClientSecret: a.ClientSecret,
			ClientUUID:   a.ClientUUID,
			Interval:     3000,
		},
	}
}

func (a *Instance) UpdateRuntimeData() {
	if a.simulator == nil {
		return
	}

	a.CPUUsage = a.simulator.GetCPUUsage()
	a.MemUsage = a.simulator.GetMemoryUsage()
	a.DiskUsage = a.simulator.GetDiskUsage()
	a.NetworkIn = a.simulator.GetNetworkInTotal()
	a.NetworkOut = a.simulator.GetNetworkOutTotal()
}

func (a *Instance) Update(req UpdateRequest) error {
	needRestart := false

	if req.ServerAddress != nil {
		a.ServerAddress = *req.ServerAddress
		needRestart = true
	}
	if req.ClientSecret != nil {
		a.ClientSecret = *req.ClientSecret
		needRestart = true
	}
	if req.ClientUUID != nil {
		a.ClientUUID = *req.ClientUUID
		needRestart = true
	}
	if req.CountryCode != nil {
		a.CountryCode = *req.CountryCode
	}
	if req.CPUModel != nil {
		a.CPUModel = *req.CPUModel
		needRestart = true
	}
	if req.CPU != nil {
		if *req.CPU <= 0 {
			return fmt.Errorf("cpu must be greater than 0")
		}
		a.CPU = *req.CPU
		needRestart = true
	}
	if req.RAM != nil {
		if *req.RAM <= 0 {
			return fmt.Errorf("ram must be greater than 0")
		}
		a.RAM = *req.RAM
		needRestart = true
	}
	if req.Swap != nil {
		a.Swap = *req.Swap
		needRestart = true
	}
	if req.DiskSize != nil {
		if *req.DiskSize <= 0 {
			return fmt.Errorf("disk_size must be greater than 0")
		}
		a.DiskSize = *req.DiskSize
		needRestart = true
	}
	if req.LoadLevel != nil {
		if *req.LoadLevel != "low" && *req.LoadLevel != "medium" && *req.LoadLevel != "high" {
			return fmt.Errorf("invalid load_level, must be one of: low, medium, high")
		}
		a.LoadLevel = *req.LoadLevel
		needRestart = true
	}
	if req.UseSSL != nil {
		a.UseSSL = *req.UseSSL
		needRestart = true
	}

	a.UpdatedAt = time.Now()

	if needRestart {
		a.Stop()
		a.ConnectionStatus = ConnectionStatus{
			Status: ConnectionStatusIdle,
		}
		a.healthCheckStop = make(chan struct{})
		a.config = a.createAgentConfig()
		a.simulator = simulator.NewDataSimulator(a.config)
		go a.Start()
	}

	return nil
}
