package agent

import (
	"context"
	"sync"
	"time"
	"virtual-nezha-agent/config"
	"virtual-nezha-agent/simulator"

	"google.golang.org/grpc"
)

const (
	ConnectionStatusIdle         = "idle"
	ConnectionStatusConnecting   = "connecting"
	ConnectionStatusConnected    = "connected"
	ConnectionStatusReconnecting = "reconnecting"
	ConnectionStatusError        = "error"
	ConnectionStatusStopped      = "stopped"
)

type ReconnectConfig struct {
	MaxRetries      int
	BaseDelay       time.Duration
	MaxDelay        time.Duration
	Multiplier      float64
	Jitter          bool
	HealthCheckFreq time.Duration
}

var DefaultReconnectConfig = ReconnectConfig{
	MaxRetries:      0,
	BaseDelay:       1 * time.Second,
	MaxDelay:        300 * time.Second,
	Multiplier:      2.0,
	Jitter:          true,
	HealthCheckFreq: 30 * time.Second,
}

type ConnectionStatus struct {
	Status        string    `json:"status"`
	LastConnected time.Time `json:"last_connected"`
	LastError     string    `json:"last_error,omitempty"`
	RetryCount    int       `json:"retry_count"`
	NextRetryAt   time.Time `json:"next_retry_at,omitempty"`
}

type Instance struct {
	ID            string    `json:"id"`
	ServerAddress string    `json:"server_address"`
	ClientSecret  string    `json:"client_secret"`
	ClientUUID    string    `json:"client_uuid"`
	CountryCode   string    `json:"country_code"`
	CPUModel      string    `json:"cpu_model"`
	CPU           int       `json:"cpu"`
	RAM           int       `json:"ram"`
	Swap          int       `json:"swap"`
	DiskSize      int       `json:"disk_size"`
	LoadLevel     string    `json:"load_level"`
	Status        string    `json:"status"`
	NetworkIn     uint64    `json:"network_in"`
	NetworkOut    uint64    `json:"network_out"`
	CPUUsage      float64   `json:"cpu_usage"`
	MemUsage      uint64    `json:"mem_usage"`
	DiskUsage     uint64    `json:"disk_usage"`
	UseSSL        bool      `json:"use_ssl"`
	CreatedAt     time.Time `json:"created_at"`
	UpdatedAt     time.Time `json:"updated_at"`

	ConnectionStatus ConnectionStatus `json:"connection_status"`

	config          *config.AgentConfig
	simulator       *simulator.DataSimulator
	cancelFunc      context.CancelFunc
	conn            *grpc.ClientConn
	mutex           sync.RWMutex
	reconnectConfig ReconnectConfig
	isReconnecting  int32
	lastHealthCheck time.Time
	healthCheckStop chan struct{}
}

type CreateRequest struct {
	ServerAddress string `json:"server_address"`
	ClientSecret  string `json:"client_secret"`
	ClientUUID    string `json:"client_uuid"`
	CountryCode   string `json:"country_code"`
	CPUModel      string `json:"cpu_model"`
	CPU           int    `json:"cpu"`
	RAM           int    `json:"ram"`
	Swap          int    `json:"swap"`
	DiskSize      int    `json:"disk_size"`
	LoadLevel     string `json:"load_level"`
	UseSSL        bool   `json:"use_ssl"`
}

type UpdateRequest struct {
	ServerAddress *string `json:"server_address,omitempty"`
	ClientSecret  *string `json:"client_secret,omitempty"`
	ClientUUID    *string `json:"client_uuid,omitempty"`
	CountryCode   *string `json:"country_code,omitempty"`
	CPUModel      *string `json:"cpu_model,omitempty"`
	CPU           *int    `json:"cpu,omitempty"`
	RAM           *int    `json:"ram,omitempty"`
	Swap          *int    `json:"swap,omitempty"`
	DiskSize      *int    `json:"disk_size,omitempty"`
	LoadLevel     *string `json:"load_level,omitempty"`
	UseSSL        *bool   `json:"use_ssl,omitempty"`
}
