// Package geoip 提供地理位置IP生成功能
// 根据国家代码生成该国家/地区的真实IP段内的随机IP地址
package geoip

import (
	"embed"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"math/rand"
	"net"
	"strings"
)

//go:embed data/*
var geoData embed.FS

type geoIp struct {
	Rules []struct {
		IpCidr []string `json:"ip_cidr"`
	} `json:"rules"`
}

var geomap map[string][]net.IPNet

func init() {
	dir, err := geoData.ReadDir("data")
	if err != nil {
		panic(err)
	}
	geomap = make(map[string][]net.IPNet)
	for _, file := range dir {
		readData, err2 := geoData.ReadFile(fmt.Sprintf("data/%s", file.Name()))
		code := strings.ReplaceAll(file.Name(), ".json", "")
		code = strings.ReplaceAll(code, "geoip-", "")
		if err2 != nil {
			continue
		}
		g := geoIp{}
		_ = json.Unmarshal(readData, &g)
		for _, rule := range g.Rules {
			for _, s := range rule.IpCidr {
				_, ipNet, err3 := net.ParseCIDR(s)
				if err3 != nil {
					continue
				}
				if ipNet.IP.To4() == nil {
					continue
				}
				geomap[code] = append(geomap[code], *ipNet)
			}
		}
	}
}
func GetIP(code string) string {
	if val, ok := geomap[strings.ToLower(code)]; ok {
		ip := val[rand.Intn(len(val))]
		return RandomIPFromIPNet(&ip).String()
	}
	return fmt.Sprintf("%d.%d.%d.%d", rand.Int31n(255), rand.Int31n(255), rand.Int31n(255), rand.Int31n(255))
}

// RandomIPFromIPNet 从给定的IPNet中随机选择一个有效的IP地址
func RandomIPFromIPNet(ipNet *net.IPNet) net.IP {
	// 获取网络地址和掩码
	mask := ipNet.Mask
	ones, bits := mask.Size()

	// 对于IPv4
	if len(mask) == net.IPv4len {
		// 计算可用的主机数量
		hostBits := bits - ones
		if hostBits <= 0 {
			return nil // 没有可用的主机地址
		}

		// 生成随机的主机部分
		randomHost := uint32(rand.Intn(1<<hostBits-1) + 1)

		// 获取网络部分的IP
		ip := ipNet.IP.To4()
		network := binary.BigEndian.Uint32(ip)

		// 组合网络和主机部分
		fullIP := network | randomHost

		// 转换回IP格式
		result := make(net.IP, net.IPv4len)
		binary.BigEndian.PutUint32(result, fullIP)
		return result
	}

	// 对于IPv6 (实现类似，但更复杂)
	if len(mask) == net.IPv6len {
		// 简化的IPv6实现 - 实际应用中可能需要更复杂的处理
		ip := make(net.IP, net.IPv6len)
		copy(ip, ipNet.IP)

		// 随机化主机部分
		for i := ones / 8; i < net.IPv6len; i++ {
			ip[i] = byte(rand.Intn(256))
		}

		// 确保不超出网络范围
		for i := 0; i < net.IPv6len; i++ {
			ip[i] = ipNet.IP[i] | (ip[i] & ^mask[i])
		}

		return ip
	}

	return nil
}
