package handler

import (
	"encoding/json"
	"net/http"
	"virtual-nezha-agent/pkg/agent"
	"virtual-nezha-agent/pkg/manager"

	"github.com/gorilla/mux"
)

type Handler struct {
	manager   *manager.Manager
	indexHTML string
}

func New(m *manager.Manager, indexHTML string) *Handler {
	return &Handler{
		manager:   m,
		indexHTML: indexHTML,
	}
}

func (h *Handler) CreateAgent(w http.ResponseWriter, r *http.Request) {
	var req agent.CreateRequest
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		http.Error(w, "Invalid JSON", http.StatusBadRequest)
		return
	}

	instance, err := h.manager.CreateAgent(req)
	if err != nil {
		sendErrorResponse(w, http.StatusBadRequest, err.Error())
		return
	}

	sendSuccessResponse(w, "Agent创建成功", instance)
}

func (h *Handler) GetAgents(w http.ResponseWriter, r *http.Request) {
	agents := h.manager.GetAllAgents()

	response := AgentListResponse{
		Agents: agents,
		Count:  len(agents),
	}

	sendSuccessResponse(w, "获取Agent列表成功", response)
}

func (h *Handler) GetAgent(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	id := vars["id"]

	instance, err := h.manager.GetAgent(id)
	if err != nil {
		sendErrorResponse(w, http.StatusNotFound, err.Error())
		return
	}

	instance.UpdateRuntimeData()

	sendSuccessResponse(w, "获取Agent成功", instance)
}

func (h *Handler) UpdateAgent(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	id := vars["id"]

	var req agent.UpdateRequest
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		http.Error(w, "Invalid JSON", http.StatusBadRequest)
		return
	}

	instance, err := h.manager.UpdateAgent(id, req)
	if err != nil {
		if err.Error() == "agent not found" {
			sendErrorResponse(w, http.StatusNotFound, err.Error())
		} else {
			sendErrorResponse(w, http.StatusBadRequest, err.Error())
		}
		return
	}

	sendSuccessResponse(w, "Agent更新成功", instance)
}

func (h *Handler) DeleteAgent(w http.ResponseWriter, r *http.Request) {
	vars := mux.Vars(r)
	id := vars["id"]

	err := h.manager.DeleteAgent(id)
	if err != nil {
		sendErrorResponse(w, http.StatusNotFound, err.Error())
		return
	}

	sendSuccessResponse(w, "Agent删除成功", nil)
}

func (h *Handler) Index(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "text/html; charset=utf-8")
	w.Write([]byte(h.indexHTML))
}

func sendSuccessResponse(w http.ResponseWriter, message string, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	response := APIResponse{
		Code:    200,
		Message: message,
		Data:    data,
	}
	json.NewEncoder(w).Encode(response)
}

func sendErrorResponse(w http.ResponseWriter, statusCode int, message string) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(statusCode)
	response := APIResponse{
		Code:    statusCode,
		Message: message,
	}
	json.NewEncoder(w).Encode(response)
}
