package manager

import (
	"encoding/json"
	"fmt"
	"os"
	"path/filepath"
	"sync"
	"virtual-nezha-agent/pkg/agent"
)

type Manager struct {
	agents      map[string]*agent.Instance
	nextID      int
	mutex       sync.RWMutex
	persistPath string
}

type PersistentData struct {
	Agents []agent.CreateRequest `json:"agents"`
	NextID int                   `json:"next_id"`
}

func New() *Manager {
	return NewWithPersist("agents.json")
}

func NewWithPersist(persistPath string) *Manager {
	m := &Manager{
		agents:      make(map[string]*agent.Instance),
		nextID:      1,
		persistPath: persistPath,
	}

	if err := m.Load(); err != nil {
		fmt.Printf("Warning: failed to load agents from %s: %v\n", persistPath, err)
	}

	return m
}

func (m *Manager) CreateAgent(req agent.CreateRequest) (*agent.Instance, error) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if _, exists := m.agents[req.ClientUUID]; exists {
		return nil, fmt.Errorf("agent with client_uuid '%s' already exists", req.ClientUUID)
	}

	instance, err := agent.NewInstance(req)
	if err != nil {
		return nil, err
	}

	m.agents[req.ClientUUID] = instance

	go instance.Start()

	if err := m.saveUnlocked(); err != nil {
		fmt.Printf("Warning: failed to save agents: %v\n", err)
	}

	return instance, nil
}

func (m *Manager) GetAgent(id string) (*agent.Instance, error) {
	m.mutex.RLock()
	defer m.mutex.RUnlock()

	instance, exists := m.agents[id]
	if !exists {
		return nil, fmt.Errorf("agent not found")
	}
	return instance, nil
}

func (m *Manager) GetAllAgents() []*agent.Instance {
	m.mutex.RLock()
	defer m.mutex.RUnlock()

	agents := make([]*agent.Instance, 0, len(m.agents))
	for _, instance := range m.agents {
		instance.UpdateRuntimeData()
		agents = append(agents, instance)
	}
	return agents
}

func (m *Manager) UpdateAgent(id string, req agent.UpdateRequest) (*agent.Instance, error) {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	instance, exists := m.agents[id]
	if !exists {
		return nil, fmt.Errorf("agent not found")
	}

	if err := instance.Update(req); err != nil {
		return nil, err
	}

	if err := m.saveUnlocked(); err != nil {
		fmt.Printf("Warning: failed to save agents: %v\n", err)
	}

	return instance, nil
}

func (m *Manager) DeleteAgent(id string) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	instance, exists := m.agents[id]
	if !exists {
		return fmt.Errorf("agent not found")
	}

	instance.Stop()
	delete(m.agents, id)

	if err := m.saveUnlocked(); err != nil {
		fmt.Printf("Warning: failed to save agents: %v\n", err)
	}

	return nil
}

func (m *Manager) Load() error {
	if m.persistPath == "" {
		return nil
	}

	data, err := os.ReadFile(m.persistPath)
	if err != nil {
		if os.IsNotExist(err) {
			return nil
		}
		return fmt.Errorf("failed to read persist file: %w", err)
	}

	var persistentData PersistentData
	if err := json.Unmarshal(data, &persistentData); err != nil {
		return fmt.Errorf("failed to unmarshal persist data: %w", err)
	}

	m.nextID = persistentData.NextID
	for _, req := range persistentData.Agents {
		instance, err := agent.NewInstance(req)
		if err != nil {
			fmt.Printf("Warning: failed to create agent %s: %v\n", req.ClientUUID, err)
			continue
		}
		m.agents[req.ClientUUID] = instance
		go instance.Start()
	}

	return nil
}

func (m *Manager) Save() error {
	if m.persistPath == "" {
		return nil
	}

	m.mutex.RLock()
	defer m.mutex.RUnlock()

	agentRequests := make([]agent.CreateRequest, 0, len(m.agents))
	for _, instance := range m.agents {
		req := agent.CreateRequest{
			ServerAddress: instance.ServerAddress,
			ClientSecret:  instance.ClientSecret,
			ClientUUID:    instance.ClientUUID,
			CountryCode:   instance.CountryCode,
			CPUModel:      instance.CPUModel,
			CPU:           instance.CPU,
			RAM:           instance.RAM,
			Swap:          instance.Swap,
			DiskSize:      instance.DiskSize,
			LoadLevel:     instance.LoadLevel,
			UseSSL:        instance.UseSSL,
		}
		agentRequests = append(agentRequests, req)
	}

	persistentData := PersistentData{
		Agents: agentRequests,
		NextID: m.nextID,
	}

	data, err := json.MarshalIndent(persistentData, "", "  ")
	if err != nil {
		return fmt.Errorf("failed to marshal persist data: %w", err)
	}

	dir := filepath.Dir(m.persistPath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		return fmt.Errorf("failed to create persist directory: %w", err)
	}

	if err := os.WriteFile(m.persistPath, data, 0644); err != nil {
		return fmt.Errorf("failed to write persist file: %w", err)
	}

	return nil
}

func (m *Manager) saveUnlocked() error {
	if m.persistPath == "" {
		return nil
	}

	agentRequests := make([]agent.CreateRequest, 0, len(m.agents))
	for _, instance := range m.agents {
		req := agent.CreateRequest{
			ServerAddress: instance.ServerAddress,
			ClientSecret:  instance.ClientSecret,
			ClientUUID:    instance.ClientUUID,
			CountryCode:   instance.CountryCode,
			CPUModel:      instance.CPUModel,
			CPU:           instance.CPU,
			RAM:           instance.RAM,
			Swap:          instance.Swap,
			DiskSize:      instance.DiskSize,
			LoadLevel:     instance.LoadLevel,
			UseSSL:        instance.UseSSL,
		}
		agentRequests = append(agentRequests, req)
	}

	persistentData := PersistentData{
		Agents: agentRequests,
		NextID: m.nextID,
	}

	data, err := json.MarshalIndent(persistentData, "", "  ")
	if err != nil {
		return fmt.Errorf("failed to marshal persist data: %w", err)
	}

	dir := filepath.Dir(m.persistPath)
	if err := os.MkdirAll(dir, 0755); err != nil {
		return fmt.Errorf("failed to create persist directory: %w", err)
	}

	if err := os.WriteFile(m.persistPath, data, 0644); err != nil {
		return fmt.Errorf("failed to write persist file: %w", err)
	}

	return nil
}
