// Package simulator 提供系统数据仿真功能
// 根据配置生成各种系统指标的模拟数据，支持多种仿真模式
package simulator

import (
	"math"
	"math/rand/v2"
	"sync"
	"time"
	"virtual-nezha-agent/config"
)

// DataSimulator 数据仿真器
type DataSimulator struct {
	config    *config.AgentConfig
	startTime time.Time
	counter   map[string]int64 // 计数器，用于线性和波形模式
	mutex     sync.RWMutex

	// 网络总量累积值
	netInTotal     uint64
	netOutTotal    uint64
	lastUpdateTime time.Time // 上次更新网络总量的时间
}

// NewDataSimulator 创建新的数据仿真器
func NewDataSimulator(config *config.AgentConfig) *DataSimulator {
	now := time.Now()
	return &DataSimulator{
		config:         config,
		startTime:      now,
		counter:        make(map[string]int64),
		netInTotal:     uint64(config.Network.InTotal.Base),
		netOutTotal:    uint64(config.Network.OutTotal.Base),
		lastUpdateTime: now,
	}
}

// simulateValueInternal 内部模拟数值方法（不加锁）
func (s *DataSimulator) simulateValueInternal(key string, c config.UsageConfig) float64 {
	counter := s.counter[key]
	s.counter[key]++

	switch c.Mode {
	case config.SimulationFixed:
		// 固定值模式：在基础值附近小幅波动
		variation := (rand.Float64() - 0.5) * c.Variance
		return c.Base + variation

	case config.SimulationLinear:
		// 线性变化模式：按步长线性增长，达到最大值后重置
		value := c.Base + float64(counter)*c.Step
		if value > c.Max {
			s.counter[key] = 0
			return c.Min
		}
		if value < c.Min {
			return c.Min
		}
		return value

	case config.SimulationRandom:
		// 范围随机模式：在最小值和最大值之间随机
		return c.Min + rand.Float64()*(c.Max-c.Min)

	case config.SimulationWave:
		// 波形变化模式：正弦波形变化
		if c.Period <= 0 {
			c.Period = 60 // 默认60秒周期
		}
		angle := 2 * math.Pi * float64(counter) / float64(c.Period)
		amplitude := (c.Max - c.Min) / 2
		midpoint := (c.Max + c.Min) / 2
		return midpoint + amplitude*math.Sin(angle)

	default:
		return c.Base
	}
}

// simulateValue 根据配置模拟数值（带锁）
func (s *DataSimulator) simulateValue(key string, config config.UsageConfig) float64 {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	return s.simulateValueInternal(key, config)
}

// simulatePercentageAsValue 模拟百分比并转换为实际值
func (s *DataSimulator) simulatePercentageAsValue(key string, config config.UsageConfig, total uint64) uint64 {
	percentage := s.simulateValue(key, config)
	if percentage < 0 {
		percentage = 0
	}
	if percentage > 100 {
		percentage = 100
	}
	return uint64(float64(total) * percentage / 100.0)
}

// GetCPUUsage 获取CPU使用率
func (s *DataSimulator) GetCPUUsage() float64 {
	return s.simulateValue("cpu_usage", s.config.CPU.Usage)
}

// GetMemoryUsage 获取内存使用量
func (s *DataSimulator) GetMemoryUsage() uint64 {
	return s.simulatePercentageAsValue("memory_usage", s.config.Memory.Usage, s.config.Memory.Total)
}

// GetDiskUsage 获取磁盘使用量
func (s *DataSimulator) GetDiskUsage() uint64 {
	return s.simulatePercentageAsValue("disk_usage", s.config.Disk.Usage, s.config.Disk.Total)
}

// GetSwapUsage 获取交换分区使用量
func (s *DataSimulator) GetSwapUsage() uint64 {
	return s.simulatePercentageAsValue("swap_usage", s.config.Swap.Usage, s.config.Swap.Total)
}

// GetNetworkInSpeed 获取网络入站速度
func (s *DataSimulator) GetNetworkInSpeed() uint64 {
	speed := s.simulateValue("net_in_speed", s.config.Network.InSpeed)
	if speed < 0 {
		speed = 0
	}
	return uint64(speed)
}

// GetNetworkOutSpeed 获取网络出站速度
func (s *DataSimulator) GetNetworkOutSpeed() uint64 {
	speed := s.simulateValue("net_out_speed", s.config.Network.OutSpeed)
	if speed < 0 {
		speed = 0
	}
	return uint64(speed)
}

// updateNetworkTotals 更新网络总量（内部方法）
func (s *DataSimulator) updateNetworkTotals() {
	now := time.Now()
	elapsed := now.Sub(s.lastUpdateTime).Seconds()

	if elapsed > 0 {
		// 获取当前网络速度（字节/秒）- 使用内部方法避免死锁
		inSpeed := s.simulateValueInternal("net_in_speed", s.config.Network.InSpeed)
		outSpeed := s.simulateValueInternal("net_out_speed", s.config.Network.OutSpeed)

		if inSpeed < 0 {
			inSpeed = 0
		}
		if outSpeed < 0 {
			outSpeed = 0
		}

		// 计算这段时间内的流量增量
		inIncrement := uint64(inSpeed * elapsed)
		outIncrement := uint64(outSpeed * elapsed)

		s.netInTotal += inIncrement
		s.netOutTotal += outIncrement

		// 更新最后更新时间
		s.lastUpdateTime = now
	}
}

// GetNetworkInTotal 获取网络入站总量
func (s *DataSimulator) GetNetworkInTotal() uint64 {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	s.updateNetworkTotals()
	return s.netInTotal
}

// GetNetworkOutTotal 获取网络出站总量
func (s *DataSimulator) GetNetworkOutTotal() uint64 {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	s.updateNetworkTotals()
	return s.netOutTotal
}

// GetLoad1 获取1分钟负载
func (s *DataSimulator) GetLoad1() float64 {
	return s.simulateValue("load1", s.config.Load.Load1)
}

// GetLoad5 获取5分钟负载
func (s *DataSimulator) GetLoad5() float64 {
	return s.simulateValue("load5", s.config.Load.Load5)
}

// GetLoad15 获取15分钟负载
func (s *DataSimulator) GetLoad15() float64 {
	return s.simulateValue("load15", s.config.Load.Load15)
}

// GetProcessCount 获取进程数量
func (s *DataSimulator) GetProcessCount() uint32 {
	count := s.simulateValue("process_count", s.config.Process.Count)
	if count < 0 {
		count = 0
	}
	return uint32(count)
}

// GetTcpConnCount 获取TCP连接数
func (s *DataSimulator) GetTcpConnCount() uint32 {
	count := s.simulateValue("tcp_conn", s.config.Process.TcpConn)
	if count < 0 {
		count = 0
	}
	return uint32(count)
}

// GetUdpConnCount 获取UDP连接数
func (s *DataSimulator) GetUdpConnCount() uint32 {
	count := s.simulateValue("udp_conn", s.config.Process.UdpConn)
	if count < 0 {
		count = 0
	}
	return uint32(count)
}

// GetGPUUsage 获取GPU使用率
func (s *DataSimulator) GetGPUUsage() []float64 {
	usage := s.simulateValue("gpu_usage", s.config.GPU.Usage)
	if usage < 0 {
		usage = 0
	}
	if usage > 100 {
		usage = 100
	}

	// 为每个GPU返回使用率
	result := make([]float64, len(s.config.GPU.Models))
	for i := range result {
		// 为每个GPU添加一些随机变化
		variation := (rand.Float64() - 0.5) * 10 // ±5%的变化
		result[i] = usage + variation
		if result[i] < 0 {
			result[i] = 0
		}
		if result[i] > 100 {
			result[i] = 100
		}
	}
	return result
}

// GetUptime 获取运行时间
func (s *DataSimulator) GetUptime() uint64 {
	return uint64(time.Since(s.startTime).Seconds())
}

// Reset 重置仿真器状态
func (s *DataSimulator) Reset() {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	s.startTime = time.Now()
	s.counter = make(map[string]int64)
	s.netInTotal = uint64(s.config.Network.InTotal.Base)
	s.netOutTotal = uint64(s.config.Network.OutTotal.Base)
}
